from pyparsing import Literal, Word, printables
from detectors.pyparsingPatterns import curlyBracesBlock, roundBracesBlock
from tokenize import generate_tokens, NL, NEWLINE, INDENT, DEDENT, untokenize
from detectors.fileTools import _getAllFiles

import re
import io

def getArgsFromKwArgs(argNames, **kwargs):
    args = {key: '' for key in argNames}
    diff = set(kwargs.keys()) - set(args.keys())
    if diff:
        raise Exception("Invalid args:", tuple(diff))
    args.update(kwargs)
    return args

class DetectorException(Exception):
    def __init__(self, message):
        super().__init__(message)

def _guessSpaceIndentation(lines):
    for line in lines:
        leadingSpaces = len(line) - len(line.lstrip(' '))
        if leadingSpaces != 0:
            return leadingSpaces
    # use indentation of 2 as a default if there are no leading spsaces in all lines
    return 2

def _preparseIndentations(text):
    lines = text.splitlines()
    # guess the equivalent of space indentations to tabs by the first space indentation
    # in order to be able to exand tabs
    spaceIndentation = _guessSpaceIndentation(lines)
    linesWithIndents = []
    indentations = []
    indentlevel = 0
    lineNumber = 0
    for line in lines:
        print("+++LINE= " + line)
        lineNumber = lineNumber + 1
        lineIndentsDedents = 0
        noTabsLine = line.expandtabs(spaceIndentation)
        leadingSpaces = len(noTabsLine) - len(noTabsLine.lstrip(' '))
        indentationsLeadingSpaces = sum(indentations)
        print("+++LeadingSpaces = " + str(leadingSpaces) + ", indents = " + str(indentations))
        if leadingSpaces > indentationsLeadingSpaces:
            indentations.append(leadingSpaces - indentationsLeadingSpaces)
            lineIndentsDedents = 1
            indentlevel = indentlevel + 1
            print("+ INDENT")
        if leadingSpaces < indentationsLeadingSpaces:
            while leadingSpaces < indentationsLeadingSpaces:
                lastIndentation = indentations[-1]
                indentations = indentations[:-1]
                indentationsLeadingSpaces = indentationsLeadingSpaces - lastIndentation
                lineIndentsDedents = lineIndentsDedents - 1
                indentlevel = indentlevel - 1
                print("+ DEDENT")
            if leadingSpaces != indentationsLeadingSpaces:
                print("INDENT LEVEL = " + str(indentlevel))
                print("+++LeadingSpaces = " + str(leadingSpaces))
                print("+++indentationsLeadingSpaces = " + str(indentationsLeadingSpaces))
                if indentlevel != 0 and leadingSpaces > indentationsLeadingSpaces:
                    # if condition above checks that we don't raise an exception if we go
                    # below the original indent, so that e.g. the following is permitted:
                    #     image: rabbitmq:3.7-management-alpine
                    #     ports:
                    #       - "5672"
                    #   payment:
                    #     build:
                    # we don't want to raise the unbalanced exception for "payment" in the example
                    raise DetectorException("unbalanced indentations in line " + str(lineNumber) + ": '" + line + "'")            
        linesWithIndents.append([lineIndentsDedents, line])
    return linesWithIndents

class DetectorContext(object):
    def __init__(self, textOrTextsPosList):
        if isinstance(textOrTextsPosList, list):
            self.textsAndPositions = textOrTextsPosList
        else:
            self.textsAndPositions= [[textOrTextsPosList, 0]]

    def _runParserDetector(self, pattern = None, matchExtractor = None, errorMessage = "", useLastTokenAsStart = False):
        newTexts = []
        for text, priorPosition in self.textsAndPositions:
            for tokens, start, end in pattern.scanString(text):
                if useLastTokenAsStart:
                    # here tokens[-1] is expected to be the start position added by the parse action
                    start = tokens[-1]
                match = matchExtractor(text, start, end)
                print("*** MATCH = " + match + "\n***")
                newTexts.append([match, priorPosition + start])

        if len(newTexts) == 0:
            raise DetectorException(errorMessage)
        return DetectorContext(newTexts)

    def containsJsFunction(self, functionName):
        return self._runParserDetector(pattern = Literal("function") + Literal(functionName) + 
                Literal("(") + Word(printables, excludeChars=")") + Literal(")") + curlyBracesBlock,
            matchExtractor = lambda text, matchStart, matchEnd: text[matchStart+1:matchEnd-1],
            errorMessage = "JS function '" + functionName + "' not found",
            useLastTokenAsStart = True)

    def containsProcCallTo(self, calledProc):
        return self._runParserDetector(pattern = Literal(calledProc) + roundBracesBlock,
            matchExtractor = lambda text, matchStart, matchEnd: text[matchStart+1:matchEnd-1],
            errorMessage = "call to '" + calledProc + "' not found",
            useLastTokenAsStart = True)

    def containsObjectCallTo(self, calledObject, calledMethod):
        return self._runParserDetector(pattern = Literal(calledObject) + Literal(".") + 
            Literal(calledMethod) + roundBracesBlock,
            matchExtractor = lambda text, matchStart, matchEnd: text[matchStart+1:matchEnd-1],
            errorMessage = "call to object'" + calledObject + "''s method '" + calledMethod + "' not found",
            useLastTokenAsStart = True)

    def matchesRegex(self, pattern):
        newTexts = []
        for text, priorPosition in self.textsAndPositions:
            if re.match(pattern, text):
                # we leave the position at 0, as the whole text is appended as a match
                newTexts.append([text, 0])
        if len(newTexts) == 0:
            raise DetectorException("regular expression '"+ pattern + "' not matched")
        return DetectorContext(newTexts)

    def matchesPattern(self, rules):
        return self._runParserDetector(pattern = rules,
            matchExtractor = lambda text, matchStart, matchEnd: text[matchStart:matchEnd],
            errorMessage = "rules '" + str(rules) + "' not found")

    # apply each rule on this detector context, return the match of 
    # the last rule as the new context
    def matchesPatterns(self, rulesList):
        result = None
        for rules in rulesList:
            result = self.matchesPattern(rules)
        return result

    def containsCurlyBracesBlock(self, rulesBeforeBlock):
        return self._runParserDetector(pattern = rulesBeforeBlock + curlyBracesBlock,
            matchExtractor = lambda text, matchStart, matchEnd: text[matchStart+1:matchEnd-1],
            errorMessage = "curly braces block starting with rules '" + str(rulesBeforeBlock) + "' not found",
            useLastTokenAsStart = True)

    # basic indented block matcher, not considering special case like indents/dedents 
    # in round braces are ignored, as in Python
    def matchIndentedBlock(self, initialPattern, errorMessage):
        newTexts = []
        for text, priorPosition in self.textsAndPositions:
            linesWithIndents = _preparseIndentations(text)
            result = ""
            indent = 0
            initialIndent = None
            firstIndentFound = False
            initialPatternFound = False
            position = -99999 # dummy init

            for indentDedent, line in linesWithIndents:
                print("### INDENT/DEDENT = " + str(indentDedent) + "    LINE= "+ line)
                indent = indent + indentDedent
                if initialIndent == None:
                    initialIndent = indent

                if not initialPatternFound:
                    if indent == initialIndent:
                        for tokens, start, end in initialPattern.scanString(line):
                            print("INITIAL PATTERN!  "+ str(indent))
                            initialPatternFound = True
                            position = start
                            break
                else:
                    if not firstIndentFound:
                        if indentDedent == 1:
                            firstIndentFound = True
                            result = result + line + "\n"
                        elif not (indentDedent == 0 and line.strip() == ""):
                            raise DetectorException("looking for start of indented block but got '" + line + "'")
                    else:
                        if indent <= initialIndent:
                            break
                        else: 
                            result = result + line + "\n"
        print("RESULT INDENTED BLOCK = " + result + "*********************************")
        newTexts.append([result, priorPosition + position])
        if len(newTexts) == 0:
             raise DetectorException(errorMessage)
        return DetectorContext(newTexts)  

    def matchIndentedPythonBlock(self, initialPattern, errorMessage):
        newTexts = []
        position = -99999 # dummy init
        for text, priorPosition in self.textsAndPositions:
            for tokens, start, end in initialPattern.scanString(text):
                print("INDENTED BLOCK MATCHER, START AT: " + str(end))
                match = text[end:]
                #print("*** MATCH = " + match)
                position = start

                result = []
                indent = 1
                firstIndentFound = False
                for token in generate_tokens(io.StringIO(match).readline):
                    tokenType = token[0]
                    tokenStart = token[2]
                    tokenEnd = token[3]
                    tokenContent = token[4]
                    #print("tokentype: " + str(tokenType) + " token = " + tokenContent)
                    if not firstIndentFound:
                        if not tokenType in [NL, NEWLINE, INDENT]:
                            raise DetectorException("looking for start of indented Python block but got '" + tokenContent + "'")
                        if tokenType == INDENT:
                            firstIndentFound = True
                    else:
                        if tokenType == INDENT:
                            indent = indent + 1
                        if tokenType == DEDENT:
                            indent = indent - 1
                    if indent <= 0:
                        #print("found end of indented block: " + str(token))
                        break
                    else:
                        result.append(token)

                print("RESULT INDENTED BLOCK = " + untokenize(result) + "*********************************")
                newTexts.append([untokenize(result), priorPosition + position])
        if len(newTexts) == 0:
            raise DetectorException(errorMessage)
        return DetectorContext(newTexts)    
        
class DirectoryDetectorContext(object):
    def __init__(self, rootDir):
        self.rootDir = rootDir
        self.result = []

    def containsJavaScript(self):
        self.result = []
        for file in _getAllFiles(self.rootDir):
            if str(file).endswith(".js"):
                self.result.append(file)
        if len(self.result) > 0:
            return True
        return False

